<?php
if (!defined('ABSPATH')) {
    exit;
}

class Transfer_Handler {
    private $main_instance;

    public function __construct($main_instance) {
        $this->main_instance = $main_instance;
    }

    public function render_page() {
        settings_errors('bcm');
        echo '<div class="wrap">';
        echo '<h2>' . esc_html($this->main_instance->__ ('批量转移', 'Bulk Transfer')) . '</h2>';

        // 获取搜索关键词（使用统一前缀避免冲突）
        $search = isset($_GET['bcm_s_transfer']) ? sanitize_text_field(wp_unslash($_GET['bcm_s_transfer'])) : '';

        // 查询匹配文章数量
        $posts_query = new WP_Query([
            'post_type' => 'post',
            'post_status' => 'any',
            's' => $search,
            'posts_per_page' => -1,
            'fields' => 'ids'
        ]);
        $total = $posts_query->found_posts;

        // === 搜索区域（顶部）===
        echo '<div class="postbox" style="padding:20px; margin-bottom:25px; background:#f9f9f9; border-radius:8px;">';
        echo '<h3 style="margin-top:0;">' . esc_html($this->main_instance->__ ('🔍 搜索文章（可选）', '🔍 Search Posts (Optional)')) . '</h3>';
        echo '<form method="get">';
        echo '<input type="hidden" name="page" value="bulk-content-manager" />';
        echo '<input type="hidden" name="tab" value="transfer" />';

        echo '<p>';
        echo '<label for="bcm_s_transfer"><strong>' . esc_html($this->main_instance->__ ('标题包含关键词：', 'Title contains:')) . '</strong></label><br>';
        echo '<input type="text" id="bcm_s_transfer" name="bcm_s_transfer" value="' . esc_attr($search) . '" placeholder="' . esc_attr($this->main_instance->__ ('输入关键词...', 'Enter keyword...')) . '" style="width:300px; padding:6px; font-size:15px;" />';
        echo '&nbsp;<input type="submit" class="button button-primary" value="' . esc_attr($this->main_instance->__ ('搜索文章', 'Search Posts')) . '" />';
        if ($search) {
            echo '&nbsp;<a href="' . esc_url(admin_url('admin.php?page=bulk-content-manager&tab=transfer')) . '" class="button">' . esc_html($this->main_instance->__ ('清除搜索', 'Clear Search')) . '</a>';
        }
        echo '</p>';

        if ($search) {
            printf(
                '<div style="margin-top:15px; padding:12px; background:#e7f4ff; border-left:4px solid #2271b1; color:#000; font-weight:bold;">%s</div>',
                sprintf(
                    $this->main_instance->__ (
                        '✅ 搜索关键词：<strong>%1$s</strong>，共匹配 <span style="color:#d63638; font-size:18px;">%2$d</span> 篇文章。<br>⚠️ <strong>本次转移操作将仅作用于这 %2$d 篇搜索结果中的文章。</strong>',
                        '✅ Keyword: <strong>%1$s</strong>, matched <span style="color:#d63638; font-size:18px;">%2$d</span> posts.<br>⚠️ <strong>This transfer will only affect these %2$d posts.</strong>'
                    ),
                    esc_html($search),
                    (int)$total
                )
            );
        } else {
            echo '<div style="margin-top:15px; padding:12px; background:#fff8e5; border-left:4px solid #dba617; color:#000;">';
            echo esc_html($this->main_instance->__ ('ℹ️ 未设置搜索条件，以下操作将影响<strong>全部文章</strong>（包含已发布、草稿、待审核、私密等所有状态）。', 'ℹ️ No search term. The following actions will affect <strong>all posts</strong> (including publish, draft, pending, private, etc.).'));
            echo '</div>';
        }
        echo '</form>';
        echo '</div>';

        // === 转移至作者表单 ===
        echo '<h3>' . esc_html($this->main_instance->__ ('1. 转移至指定作者', '1. Transfer to Author')) . '</h3>';
        echo '<form method="post" action="">';
        wp_nonce_field('bulk-transfer-author', '_wpnonce_bulk_transfer_author');
        echo '<input type="hidden" name="page" value="bulk-content-manager" />';
        echo '<input type="hidden" name="tab" value="transfer" />';
        if ($search) echo '<input type="hidden" name="bcm_s_transfer" value="' . esc_attr($search) . '" />';

        $users = get_users(['who' => 'authors', 'orderby' => 'display_name']);
        if ($users) {
            echo '<select name="new_author" required style="min-width:200px; margin:10px 0;">';
            echo '<option value="">' . esc_html($this->main_instance->__ ('— 请选择作者 —', '— Select Author —')) . '</option>';
            foreach ($users as $user) {
                printf(
                    '<option value="%d">%s (%s)</option>',
                    esc_attr($user->ID),
                    esc_html($user->display_name),
                    esc_html($user->user_login)
                );
            }
            echo '</select>';
            echo '<input type="submit" name="do_transfer_author" class="button button-primary" value="' . esc_attr($this->main_instance->__ ('转移至该作者', 'Transfer to Author')) . '" onclick="return confirmTransfer(' . (int)$total . ');" />';
        } else {
            echo '<p>' . esc_html($this->main_instance->__ ('暂无可用作者。', 'No authors available.')) . '</p>';
        }
        echo '</form>';

        echo '<hr style="margin:30px 0;">';

        // === 转移至分类表单 ===
        echo '<h3>' . esc_html($this->main_instance->__ ('2. 转移至指定分类', '2. Transfer to Category')) . '</h3>';
        echo '<form method="post" action="">';
        wp_nonce_field('bulk-transfer-category', '_wpnonce_bulk_transfer_category');
        echo '<input type="hidden" name="page" value="bulk-content-manager" />';
        echo '<input type="hidden" name="tab" value="transfer" />';
        if ($search) echo '<input type="hidden" name="bcm_s_transfer" value="' . esc_attr($search) . '" />';

        $categories = get_categories(['hide_empty' => false]);
        if ($categories) {
            echo '<select name="new_category" required style="min-width:200px; margin:10px 0;">';
            echo '<option value="">' . esc_html($this->main_instance->__ ('— 请选择分类 —', '— Select Category —')) . '</option>';
            foreach ($categories as $cat) {
                printf(
                    '<option value="%d">%s (%d)</option>',
                    esc_attr($cat->term_id),
                    esc_html($cat->name),
                    (int)$cat->count
                );
            }
            echo '</select>';
            echo '<input type="submit" name="do_transfer_category" class="button button-primary" value="' . esc_attr($this->main_instance->__ ('转移至该分类', 'Transfer to Category')) . '" onclick="return confirmTransfer(' . (int)$total . ');" />';
        } else {
            echo '<p>' . esc_html($this->main_instance->__ ('暂无分类。', 'No categories available.')) . '</p>';
        }
        echo '</form>';

        echo '</div>'; // .wrap

        ?>
        <script>
        function confirmTransfer(count) {
            if (count === 0) {
                alert('<?php echo esc_js($this->main_instance->__ ('没有匹配的文章，无法执行操作。', 'No posts matched. Cannot proceed.')); ?>');
                return false;
            }
            return confirm('<?php echo esc_js($this->main_instance->__ ('确定要对 %d 篇文章执行此操作吗？\n\n⚠️ 此操作不可逆！', 'Confirm transfer for %d posts?\n\n⚠️ This action cannot be undone!')); ?>'.replace('%d', count));
        }
        </script>
        <?php
    }

    public function handle_transfer() {
        // 处理作者转移
        if (isset($_POST['do_transfer_author'])) {
            if (!isset($_POST['_wpnonce_bulk_transfer_author']) || !wp_verify_nonce($_POST['_wpnonce_bulk_transfer_author'], 'bulk-transfer-author')) {
                wp_die('Security check failed.');
            }

            $search = isset($_POST['bcm_s_transfer']) ? sanitize_text_field($_POST['bcm_s_transfer']) : '';
            $new_author = intval($_POST['new_author']);

            if (!$new_author || !get_user_by('id', $new_author)) {
                add_settings_error('bcm', 'invalid_author', $this->main_instance->__ ('无效的作者 ID。', 'Invalid author ID.'), 'error');
                wp_redirect(wp_get_referer());
                exit;
            }

            $posts = get_posts([
                'post_type' => 'post',
                'post_status' => 'any',
                's' => $search,
                'numberposts' => -1,
                'fields' => 'ids'
            ]);

            if (empty($posts)) {
                add_settings_error('bcm', 'no_posts', $this->main_instance->__ ('没有匹配的文章。', 'No posts matched.'), 'error');
                wp_redirect(wp_get_referer());
                exit;
            }

            $start_time = microtime(true);
            $count = 0;
            foreach ($posts as $post_id) {
                $result = wp_update_post(['ID' => $post_id, 'post_author' => $new_author], true);
                if (!is_wp_error($result)) $count++;
            }
            $duration = microtime(true) - $start_time;

            $this->main_instance->log_operation(
                'transfer_author',
                "New Author ID: {$new_author}, Search: '{$search}', Count: {$count}",
                true,
                '已完成',
                $duration
            );

            add_settings_error('bcm', 'success', sprintf($this->main_instance->__ ('成功转移 %d 篇文章至指定作者。', 'Successfully transferred %d posts to the author.'), $count), 'updated');
            wp_redirect(remove_query_arg(['do_transfer_author', '_wpnonce_bulk_transfer_author']));
            exit;
        }

        // 处理分类转移
        if (isset($_POST['do_transfer_category'])) {
            if (!isset($_POST['_wpnonce_bulk_transfer_category']) || !wp_verify_nonce($_POST['_wpnonce_bulk_transfer_category'], 'bulk-transfer-category')) {
                wp_die('Security check failed.');
            }

            $search = isset($_POST['bcm_s_transfer']) ? sanitize_text_field($_POST['bcm_s_transfer']) : '';
            $new_category = intval($_POST['new_category']);

            if (!$new_category || !term_exists($new_category, 'category')) {
                add_settings_error('bcm', 'invalid_category', $this->main_instance->__ ('无效的分类 ID。', 'Invalid category ID.'), 'error');
                wp_redirect(wp_get_referer());
                exit;
            }

            $posts = get_posts([
                'post_type' => 'post',
                'post_status' => 'any',
                's' => $search,
                'numberposts' => -1,
                'fields' => 'ids'
            ]);

            if (empty($posts)) {
                add_settings_error('bcm', 'no_posts', $this->main_instance->__ ('没有匹配的文章。', 'No posts matched.'), 'error');
                wp_redirect(wp_get_referer());
                exit;
            }

            $start_time = microtime(true);
            $count = 0;
            foreach ($posts as $post_id) {
                $result = wp_set_post_categories($post_id, [$new_category], false); // 替换
                if (!is_wp_error($result)) $count++;
            }
            $duration = microtime(true) - $start_time;

            $this->main_instance->log_operation(
                'transfer_category',
                "New Category ID: {$new_category}, Search: '{$search}', Count: {$count}",
                true,
                '已完成',
                $duration
            );

            add_settings_error('bcm', 'success', sprintf($this->main_instance->__ ('成功转移 %d 篇文章至指定分类。', 'Successfully transferred %d posts to the category.'), $count), 'updated');
            wp_redirect(remove_query_arg(['do_transfer_category', '_wpnonce_bulk_transfer_category']));
            exit;
        }
    }
}