<?php
if (!defined('ABSPATH')) { exit; }

class Tags_Handler {
    private $main_instance;
    private $per_page = 25;

    public function __construct($main_instance) {
        $this->main_instance = $main_instance;
    }

    public function render_page() {
        settings_errors('bcm');
        echo '<div class="wrap">';
        echo '<h2>' . esc_html($this->main_instance->__ ('标签管理', 'Tags Management')) . '</h2>';
        $this->render_import_export_section();

        $paged = max(1, intval($_GET['paged'] ?? 1));
        $offset = ($paged - 1) * $this->per_page;

        $total_tags = wp_count_terms('post_tag', ['hide_empty' => false]);
        $total_pages = ceil($total_tags / $this->per_page);

        $tags = get_terms([
            'taxonomy' => 'post_tag',
            'hide_empty' => false,
            'orderby' => 'name',
            'number' => $this->per_page,
            'offset' => $offset
        ]);

        echo '<form method="post" action="">';
        wp_nonce_field('bulk-tags', '_wpnonce_bulk_tags');
        echo '<input type="hidden" name="page" value="bulk-content-manager" />';
        echo '<input type="hidden" name="tab" value="tags" />';

        $this->display_tablenav('top', $total_pages, $paged, $total_tags);

        echo '<table class="wp-list-table widefat fixed striped posts">';
        echo '<thead><tr>';
        echo '<td class="manage-column column-cb check-column"><label class="screen-reader-text" for="cb-select-all-1">全选</label><input id="cb-select-all-1" type="checkbox" /></td>';
        echo '<th>' . esc_html($this->main_instance->__ ('ID', 'ID')) . '</th>';
        echo '<th>' . esc_html($this->main_instance->__ ('名称', 'Name')) . '</th>';
        echo '<th>' . esc_html($this->main_instance->__ ('别名', 'Slug')) . '</th>';
        echo '<th>' . esc_html($this->main_instance->__ ('文章数', 'Posts')) . '</th>';
        echo '</tr></thead>';
        echo '<tbody>';

        if (empty($tags)) {
            echo '<tr><td colspan="5">' . esc_html($this->main_instance->__ ('未找到标签。', 'No tags found.')) . '</td></tr>';
        } else {
            foreach ($tags as $tag) {
                echo '<tr>';
                echo '<th class="check-column"><input type="checkbox" name="tag[]" value="' . esc_attr($tag->term_id) . '" /></th>';
                echo '<td>' . esc_html($tag->term_id) . '</td>';
                echo '<td>';
                echo '<strong>' . esc_html($tag->name) . '</strong><br>';
                echo '<a href="#" class="edit-tag" data-id="' . esc_attr($tag->term_id) . '">' . esc_html($this->main_instance->__ ('快速编辑', 'Quick Edit')) . '</a>';
                echo '</td>';
                echo '<td>' . esc_html($tag->slug) . '</td>';
                // 🔧 修复：确保 tag 参数正确传递（无需 urlencode）
                $posts_link = add_query_arg([
                    'page' => 'bulk-content-manager',
                    'tab' => 'posts',
                    'tag' => $tag->slug
                ], admin_url('admin.php'));
                echo '<td><a href="' . esc_url($posts_link) . '">' . esc_html($tag->count) . '</a></td>';
                echo '</tr>';
            }
        }

        echo '</tbody></table>';
        $this->display_tablenav('bottom', $total_pages, $paged, $total_tags);
        echo '</form>';

        echo '<div id="edit-tag-modal" style="display:none; position:fixed; top:20%; left:50%; transform:translateX(-50%); background:white; padding:20px; border:1px solid #ccc; z-index:9999; box-shadow:0 0 10px rgba(0,0,0,0.3);">';
        echo '<h3>' . esc_html($this->main_instance->__ ('编辑标签名称', 'Edit Tag Name')) . '</h3>';
        echo '<form id="edit-tag-form">';
        echo '<input type="hidden" id="edit-tag-id" name="id" />';
        echo '<input type="text" id="edit-tag-name" name="name" style="width:100%; margin:10px 0;" required />';
        echo '<button type="submit" class="button button-primary">' . esc_html($this->main_instance->__ ('保存', 'Save')) . '</button>';
        echo ' <button type="button" class="button" onclick="document.getElementById(\'edit-tag-modal\').style.display=\'none\'">' . esc_html($this->main_instance->__ ('取消', 'Cancel')) . '</button>';
        echo '</form>';
        echo '</div>';
        echo '</div>';
    }

    private function render_import_export_section() {
        echo '<div class="bcm-import-export">';
        echo '<strong>' . esc_html($this->main_instance->__ ('导入/导出', 'Import / Export')) . '</strong><br><br>';
        echo '<a href="' . esc_url($this->main_instance->get_export_url('tags')) . '" class="button button-secondary">' . esc_html($this->main_instance->__ ('导出所有标签', 'Export All Tags')) . '</a> ';
        echo '<form method="post" enctype="multipart/form-data" style="display:inline-block; vertical-align:middle;">';
        wp_nonce_field('bcm_import_tags', 'bcm_nonce');
        echo '<input type="file" name="csv_file" accept=".csv" required />';
        echo '<input type="submit" name="import_tags" class="button button-primary" value="' . esc_attr($this->main_instance->__ ('从 CSV 导入标签', 'Import Tags from CSV')) . '" />';
        echo '</form>';
        echo '</div>';
    }

    private function display_tablenav($which, $total_pages = 1, $current_page = 1, $total_items = 0) {
        $actions = ['delete' => $this->main_instance->__ ('删除', 'Delete')];
        echo '<div class="tablenav ' . esc_attr($which) . '">';
        echo '<div class="alignleft actions bulkactions">';
        echo '<select name="action" id="bulk-action-selector-' . esc_attr($which) . '">';
        echo '<option value="-1">' . $this->main_instance->__ ('批量操作', 'Bulk Actions') . '</option>';
        foreach ($actions as $value => $label) {
            echo '<option value="' . esc_attr($value) . '">' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '<input type="submit" name="doaction" class="button action" value="' . esc_attr($this->main_instance->__ ('应用', 'Apply')) . '" />';
        echo '</div>';

        if ($total_pages > 1) {
            $base_url = remove_query_arg(['paged']);
            $pagination = paginate_links([
                'base' => $base_url . '%_%',
                'format' => '&paged=%#%',
                'current' => $current_page,
                'total' => $total_pages,
                'prev_text' => '&laquo;',
                'next_text' => '&raquo;',
                'type' => 'plain',
            ]);
            echo '<div class="tablenav-pages">';
            echo '<span class="displaying-num">' . sprintf(
                $this->main_instance->__ ('共 %s 项', 'Total %s items'),
                number_format_i18n($total_items)
            ) . '</span>';
            echo $pagination;
            echo '</div>';
        }
        echo '</div>';
    }

    public function handle_bulk_actions() {
        if (!isset($_POST['_wpnonce_bulk_tags']) || !wp_verify_nonce($_POST['_wpnonce_bulk_tags'], 'bulk-tags')) {
            wp_die('Security check failed.');
        }
        $action = sanitize_text_field($_POST['action']);
        $tag_ids = isset($_POST['tag']) ? array_map('intval', $_POST['tag']) : [];
        if (empty($tag_ids)) {
            add_settings_error('bcm', 'no_items', $this->main_instance->__ ('未选择任何标签。', 'No tags selected.'), 'error');
            wp_redirect(wp_get_referer() ?: admin_url('admin.php?page=bulk-content-manager&tab=tags'));
            exit;
        }
        $start_time = microtime(true);
        $count = 0;
        $errors = 0;
        foreach ($tag_ids as $tag_id) {
            if (!term_exists($tag_id, 'post_tag')) continue;
            if ($action === 'delete') {
                if (wp_delete_term($tag_id, 'post_tag')) $count++; else $errors++;
            }
        }
        $duration = microtime(true) - $start_time;
        $status = $errors > 0 ? '部分失败' : '已完成';
        $details = "Action: {$action}, Count: {$count}, Errors: {$errors}";
        $this->main_instance->log_operation("bulk_{$action}_tags", $details, $errors === 0, $status, $duration);
        if ($errors === 0) {
            add_settings_error('bcm', 'success', sprintf($this->main_instance->__ ('成功处理 %d 个标签。', 'Successfully processed %d tags.'), $count), 'updated');
        } else {
            add_settings_error('bcm', 'partial', sprintf($this->main_instance->__ ('处理完成，但有 %d 个错误。', 'Processing complete, but with %d errors.'), $errors), 'error');
        }
        wp_redirect(remove_query_arg(['action', 'tag', '_wpnonce_bulk_tags']));
        exit;
    }
}