<?php
if (!defined('ABSPATH')) {
    exit;
}

class Posts_Handler {
    private $main_instance;
    private $per_page = 25;

    public function __construct($main_instance) {
        $this->main_instance = $main_instance;
    }

    public function render_page() {
        settings_errors('bcm');
        echo '<div class="wrap">';
        echo '<h2>' . esc_html($this->main_instance->__ ('文章管理', 'Posts Management')) . '</h2>';
        $this->render_import_export_section();

        // 获取筛选参数（统一前缀 bcm_ 避免冲突）
        $selected_cat = isset($_GET['bcm_cat']) ? (int)$_GET['bcm_cat'] : 0;
        $selected_tag = isset($_GET['bcm_tag']) ? sanitize_text_field(wp_unslash($_GET['bcm_tag'])) : '';
        $search_title = isset($_GET['bcm_s']) ? sanitize_text_field(wp_unslash($_GET['bcm_s'])) : '';

        // === 加载 Select2 CSS/JS（仅在此页面）===
        wp_enqueue_style('select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css');
        wp_enqueue_script('select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js', ['jquery'], null, true);

        // === 筛选表单 ===
        echo '<div class="postbox" style="padding:15px; margin-bottom:20px; background:#f9f9f9;">';
        echo '<h3 style="margin-top:0;">筛选文章</h3>';
        echo '<form method="get" id="bcm-filter-form">';
        echo '<input type="hidden" name="page" value="bulk-content-manager" />';
        echo '<input type="hidden" name="tab" value="posts" />';

        // 分类（Select2）
        echo '<label for="bcm_cat">分类：</label>';
        echo '<select name="bcm_cat" id="bcm_cat" style="width:200px;">';
        echo '<option value="">— 全部分类 —</option>';
        $categories = get_categories(['hide_empty' => false, 'orderby' => 'name']);
        foreach ($categories as $cat) {
            printf(
                '<option value="%d" %s>%s (%d)</option>',
                esc_attr($cat->term_id),
                selected($selected_cat, $cat->term_id, false),
                esc_html($cat->name),
                (int)$cat->count
            );
        }
        echo '</select> &nbsp; ';

        // 标签（Select2）
        echo '<label for="bcm_tag">标签：</label>';
        echo '<select name="bcm_tag" id="bcm_tag" style="width:200px;">';
        echo '<option value="">— 全部标签 —</option>';
        $tags = get_terms([
            'taxonomy'   => 'post_tag',
            'hide_empty' => false,
            'orderby'    => 'name',
            'number'     => 300
        ]);
        if (!is_wp_error($tags)) {
            foreach ($tags as $tag) {
                printf(
                    '<option value="%s" %s>%s (%d)</option>',
                    esc_attr($tag->slug),
                    selected($selected_tag, $tag->slug, false),
                    esc_html($tag->name),
                    (int)$tag->count
                );
            }
        }
        echo '</select> &nbsp; ';

        // 标题搜索
        echo '<label for="bcm_s">标题包含：</label>';
        echo '<input type="text" name="bcm_s" id="bcm_s" value="' . esc_attr($search_title) . '" placeholder="输入关键词" style="width:150px;" /> &nbsp; ';

        echo '<input type="submit" class="button button-primary" value="筛选" />';
        if ($selected_cat || $selected_tag || $search_title) {
            echo ' <a href="' . esc_url(admin_url('admin.php?page=bulk-content-manager&tab=posts')) . '" class="button">清除筛选</a>';
        }
        echo '</form>';
        echo '</div>';

        // === 构建查询 ===
        $paged = max(1, intval($_GET['paged'] ?? 1));

        $args = [
            'post_type'      => 'post',
            'post_status'    => ['publish', 'draft', 'pending', 'future', 'private'],
            'posts_per_page' => $this->per_page,
            'paged'          => $paged,
            'orderby'        => 'ID',
            'order'          => 'DESC'
        ];

        $meta_query = [];
        $tax_queries = [];

        // 分类
        if ($selected_cat > 0 && term_exists($selected_cat, 'category')) {
            $tax_queries[] = [
                'taxonomy' => 'category',
                'field'    => 'term_id',
                'terms'    => $selected_cat
            ];
        }

        // 标签
        if (!empty($selected_tag) && term_exists($selected_tag, 'post_tag', 'slug')) {
            $tax_queries[] = [
                'taxonomy' => 'post_tag',
                'field'    => 'slug',
                'terms'    => $selected_tag
            ];
        }

        if (!empty($tax_queries)) {
            $args['tax_query'] = count($tax_queries) === 1 ? $tax_queries : array_merge(['relation' => 'AND'], $tax_queries);
        }

        // === 关键修复：自定义标题模糊搜索（绕过 WP 的 s 参数）===
        if (!empty($search_title)) {
            add_filter('posts_where', [$this, 'title_like_filter'], 10, 2);
            // 存储关键词到类属性，供 filter 使用
            $this->current_search = $search_title;
        }

        $query = new WP_Query($args);

        // 移除临时 filter
        if (!empty($search_title)) {
            remove_filter('posts_where', [$this, 'title_like_filter'], 10);
            unset($this->current_search);
        }

        $posts = $query->posts;
        $total_items = $query->found_posts;
        $total_pages = ceil($total_items / $this->per_page);

        // === 批量操作表单 ===
        echo '<form method="post" action="">';
        wp_nonce_field('bulk-posts', '_wpnonce_bulk_posts');
        echo '<input type="hidden" name="page" value="bulk-content-manager" />';
        echo '<input type="hidden" name="tab" value="posts" />';
        if ($selected_cat) echo '<input type="hidden" name="bcm_cat" value="' . esc_attr($selected_cat) . '" />';
        if ($selected_tag) echo '<input type="hidden" name="bcm_tag" value="' . esc_attr($selected_tag) . '" />';
        if ($search_title) echo '<input type="hidden" name="bcm_s" value="' . esc_attr($search_title) . '" />';

        $this->display_tablenav('top', $total_pages, $paged, $search_title, $total_items);

        // 文章表格
        echo '<table class="wp-list-table widefat fixed striped posts">';
        echo '<thead><tr>';
        echo '<td class="manage-column column-cb check-column"><label class="screen-reader-text" for="cb-select-all-1">全选</label><input id="cb-select-all-1" type="checkbox" /></td>';
        echo '<th>ID</th><th>标题</th><th>状态</th><th>分类</th><th>标签</th><th>日期</th>';
        echo '</tr></thead><tbody>';

        if (empty($posts)) {
            echo '<tr><td colspan="6">未找到文章。</td></tr>';
        } else {
            foreach ($posts as $post) {
                $cats = wp_get_post_categories($post->ID, ['fields' => 'names']);
                $tags = wp_get_post_tags($post->ID, ['fields' => 'names']);
                echo '<tr>';
                echo '<th class="check-column"><input type="checkbox" name="post[]" value="' . esc_attr($post->ID) . '" /></th>';
                echo '<td>' . esc_html($post->ID) . '</td>';
                echo '<td><strong>' . esc_html($post->post_title) . '</strong><br>';
                echo '<a href="#" class="edit-title" data-id="' . esc_attr($post->ID) . '">快速编辑</a></td>';
                echo '<td>' . esc_html($post->post_status) . '</td>';
                echo '<td>' . esc_html(implode(', ', $cats ?: [])) . '</td>';
                echo '<td>' . esc_html(implode(', ', $tags ?: [])) . '</td>';
                echo '<td>' . esc_html(mysql2date(get_option('date_format') . ' ' . get_option('time_format'), $post->post_date)) . '</td>';
                echo '</tr>';
            }
        }
        echo '</tbody></table>';

        $this->display_tablenav('bottom', $total_pages, $paged, $search_title, $total_items);
        echo '</form>';

        // 快速编辑模态框（略）
        echo '<div id="edit-title-modal" style="display:none; position:fixed; top:20%; left:50%; transform:translateX(-50%); background:white; padding:20px; border:1px solid #ccc; z-index:9999; box-shadow:0 0 10px rgba(0,0,0,0.3);">';
        echo '<h3>编辑文章标题</h3>';
        echo '<form id="edit-title-form">';
        echo '<input type="hidden" id="edit-post-id" name="id" />';
        echo '<input type="text" id="edit-post-title" name="title" style="width:100%; margin:10px 0;" required />';
        echo '<button type="submit" class="button button-primary">保存</button>';
        echo ' <button type="button" class="button" onclick="document.getElementById(\'edit-title-modal\').style.display=\'none\'">取消</button>';
        echo '</form>';
        echo '</div>';

        // JS：初始化 Select2 + 快速编辑
        ?>
        <script>
        jQuery(document).ready(function($) {
            $('#bcm_cat, #bcm_tag').select2({
                placeholder: "选择或搜索...",
                allowClear: true,
                width: 'resolve'
            });

            $('.edit-title').on('click', function(e) {
                e.preventDefault();
                var id = $(this).data('id');
                var title = $(this).closest('tr').find('strong').text();
                $('#edit-post-id').val(id);
                $('#edit-post-title').val(title);
                $('#edit-title-modal').show();
            });

            $('#edit-title-form').on('submit', function(e) {
                e.preventDefault();
                var formData = new FormData();
                formData.append('action', 'bcm_quick_edit_title');
                formData.append('id', $('#edit-post-id').val());
                formData.append('title', $('#edit-post-title').val());
                formData.append('_ajax_nonce', '<?php echo wp_create_nonce("bcm_quick_edit"); ?>');

                fetch(ajaxurl, {method: 'POST', body: formData})
                    .then(r => r.json())
                    .then(data => {
                        if (data.success) {
                            alert('标题已更新！');
                            location.reload();
                        } else {
                            alert('失败：' + (data.data?.message || ''));
                        }
                    });
            });
        });
        </script>
        <?php

        echo '</div>'; // .wrap
    }

    // === 新增：自定义标题模糊搜索 ===
    public function title_like_filter($where, $wp_query) {
        global $wpdb;
        if (!empty($this->current_search)) {
            $keyword = $wpdb->esc_like($this->current_search);
            $where .= $wpdb->prepare(" AND {$wpdb->posts}.post_title LIKE %s", '%' . $keyword . '%');
        }
        return $where;
    }

    // ===== 以下方法保持不变（仅微调分页参数）=====

    private function render_import_export_section() {
        echo '<div class="bcm-import-export" style="margin-bottom:20px;">';
        echo '<strong>导入/导出</strong><br><br>';
        echo '<a href="' . esc_url($this->main_instance->get_export_url('posts')) . '" class="button button-secondary">导出所有文章</a> ';
        echo '<form method="post" enctype="multipart/form-data" style="display:inline-block; vertical-align:middle;">';
        wp_nonce_field('bcm_import_posts', 'bcm_nonce');
        echo '<input type="file" name="csv_file" accept=".csv" required />';
        echo '<input type="submit" name="import_posts" class="button button-primary" value="从 CSV 导入文章" />';
        echo '</form>';
        echo '</div>';
    }

    private function display_tablenav($which, $total_pages = 1, $current_page = 1, $search = '', $total_items = 0) {
        $categories = get_categories(['hide_empty' => false, 'orderby' => 'name']);
        $authors = get_users(['who' => 'authors']);
        echo '<div class="tablenav ' . esc_attr($which) . '">';
        echo '<div class="alignleft actions bulkactions">';
        echo '<select name="action" id="bulk-action-selector-' . esc_attr($which) . '">';
        echo '<option value="-1">批量操作</option>';
        echo '<option value="delete">删除</option>';
        echo '<option value="publish">设为发布</option>';
        echo '<option value="draft">设为草稿</option>';
        echo '<option value="move_cat">移动到分类</option>';
        echo '<option value="change_author">更改作者</option>';
        echo '</select>';

        if (!empty($categories)) {
            echo '<select name="target_category" style="display:none;" id="target-cat-' . esc_attr($which) . '">';
            foreach ($categories as $cat) {
                echo '<option value="' . esc_attr($cat->term_id) . '">' . esc_html($cat->name) . '</option>';
            }
            echo '</select>';
        }

        if (!empty($authors)) {
            echo '<select name="target_author" style="display:none;" id="target-author-' . esc_attr($which) . '">';
            foreach ($authors as $author) {
                echo '<option value="' . esc_attr($author->ID) . '">' . esc_html($author->display_name) . '</option>';
            }
            echo '</select>';
        }

        echo '<input type="submit" name="doaction" class="button action" value="应用" />';
        echo '</div>';

        if ($total_pages > 1) {
            $base_url = admin_url('admin.php');
            $params = [
                'page' => 'bulk-content-manager',
                'tab'  => 'posts'
            ];
            if (isset($_GET['bcm_cat'])) $params['bcm_cat'] = (int)$_GET['bcm_cat'];
            if (isset($_GET['bcm_tag'])) $params['bcm_tag'] = sanitize_text_field($_GET['bcm_tag']);
            if (isset($_GET['bcm_s']))   $params['bcm_s']   = sanitize_text_field($_GET['bcm_s']);

            $base_url = add_query_arg($params, $base_url);

            $pagination = paginate_links([
                'base'      => $base_url . '&paged=%#%',
                'format'    => '',
                'current'   => $current_page,
                'total'     => $total_pages,
                'prev_text' => '&laquo;',
                'next_text' => '&raquo;',
            ]);

            echo '<div class="tablenav-pages">';
            echo '<span class="displaying-num">共 ' . number_format_i18n($total_items) . ' 项</span>';
            echo $pagination;
            echo '</div>';
        }
        echo '</div>';

        ?>
        <script>
        jQuery(document).ready(function($) {
            var selector = $('#bulk-action-selector-<?php echo esc_js($which); ?>');
            var catSelect = $('#target-cat-<?php echo esc_js($which); ?>');
            var authorSelect = $('#target-author-<?php echo esc_js($which); ?>');
            selector.on('change', function() {
                catSelect.toggle(this.value === 'move_cat');
                authorSelect.toggle(this.value === 'change_author');
            });
        });
        </script>
        <?php
    }

    public function handle_bulk_actions() {
        if (!isset($_POST['_wpnonce_bulk_posts']) || !wp_verify_nonce($_POST['_wpnonce_bulk_posts'], 'bulk-posts')) {
            wp_die('Security check failed.');
        }
        $action = sanitize_text_field($_POST['action']);
        $post_ids = isset($_POST['post']) ? array_map('intval', $_POST['post']) : [];
        if (empty($post_ids)) {
            add_settings_error('bcm', 'no_items', '未选择任何文章。', 'error');
            wp_redirect(wp_get_referer() ?: admin_url('admin.php?page=bulk-content-manager&tab=posts'));
            exit;
        }

        $count = 0;
        $errors = 0;
        foreach ($post_ids as $post_id) {
            if (!get_post($post_id)) continue;
            switch ($action) {
                case 'delete':
                    if (wp_delete_post($post_id, true)) $count++; else $errors++;
                    break;
                case 'publish':
                    $updated = wp_update_post(['ID' => $post_id, 'post_status' => 'publish']);
                    if ($updated && !is_wp_error($updated)) $count++; else $errors++;
                    break;
                case 'draft':
                    $updated = wp_update_post(['ID' => $post_id, 'post_status' => 'draft']);
                    if ($updated && !is_wp_error($updated)) $count++; else $errors++;
                    break;
                case 'move_cat':
                    $target_cat = isset($_POST['target_category']) ? intval($_POST['target_category']) : 0;
                    if ($target_cat && term_exists($target_cat, 'category')) {
                        $result = wp_set_post_terms($post_id, [$target_cat], 'category');
                        if (!is_wp_error($result)) $count++; else $errors++;
                    } else {
                        $errors++;
                    }
                    break;
                case 'change_author':
                    $target_author = isset($_POST['target_author']) ? intval($_POST['target_author']) : 0;
                    if ($target_author && get_user_by('id', $target_author)) {
                        $updated = wp_update_post(['ID' => $post_id, 'post_author' => $target_author]);
                        if ($updated && !is_wp_error($updated)) $count++; else $errors++;
                    } else {
                        $errors++;
                    }
                    break;
            }
        }

        if ($errors === 0) {
            add_settings_error('bcm', 'success', "成功处理 {$count} 篇文章。", 'updated');
        } else {
            add_settings_error('bcm', 'partial', "处理完成，但有 {$errors} 个错误。", 'error');
        }

        wp_redirect(remove_query_arg(['action', 'post', '_wpnonce_bulk_posts']));
        exit;
    }
}