<?php
if (!defined('ABSPATH')) { exit; }

class Categories_Handler {
    private $main_instance;

    public function __construct($main_instance) {
        $this->main_instance = $main_instance;
    }

    public function render_page() {
        // 获取所有分类（用于批量操作下拉）
        $all_cats = get_categories(['hide_empty' => false, 'orderby' => 'name']);

        echo '<div class="wrap">';
        echo '<h2>' . esc_html($this->main_instance->__ ('分类管理', 'Categories')) . '</h2>';

        // 导入/导出区域
        echo '<div class="bcm-import-export">';
        echo '<strong>' . esc_html($this->main_instance->__ ('导入/导出', 'Import/Export')) . '</strong><br><br>';
        // 导出按钮
        echo '<a href="' . esc_url($this->main_instance->get_export_url('categories')) . '" class="button button-secondary">' . esc_html($this->main_instance->__ ('导出分类 CSV', 'Export Categories CSV')) . '</a>&nbsp;&nbsp;';
        // 导入表单
        echo '<form method="post" enctype="multipart/form-data" style="display:inline-block;">';
        wp_nonce_field('bcm_import_categories', 'bcm_nonce');
        echo '<input type="file" name="csv_file" accept=".csv" required />';
        echo '<input type="submit" name="import_categories" class="button button-primary" value="' . esc_attr($this->main_instance->__ ('从 CSV 导入', 'Import from CSV')) . '" />';
        echo '</form>';
        echo '</div>';

        // 批量操作表单
        echo '<form method="post" id="categories-form">';

        // 表头（顶部）
        $this->display_tablenav('top', $all_cats);

        // 分类列表
        echo '<table class="wp-list-table widefat fixed striped posts">';
        echo '<thead><tr>';
        echo '<td class="manage-column column-cb check-column"><input type="checkbox" /></td>';
        echo '<th>' . esc_html($this->main_instance->__ ('名称', 'Name')) . '</th>';
        echo '<th>' . esc_html($this->main_instance->__ ('ID', 'ID')) . '</th>'; // ✅ 新增 ID 列
        echo '<th>' . esc_html($this->main_instance->__ ('别名', 'Slug')) . '</th>';
        echo '<th>' . esc_html($this->main_instance->__ ('文章数', 'Count')) . '</th>';
        echo '<th>' . esc_html($this->main_instance->__ ('父级', 'Parent')) . '</th>';
        echo '<th>' . esc_html($this->main_instance->__ ('描述', 'Description')) . '</th>'; // ✅ 新增 描述 列
        echo '<th>' . esc_html($this->main_instance->__ ('操作', 'Actions')) . '</th>';
        echo '</tr></thead>';
        echo '<tbody>';

        $cats = get_categories(['hide_empty' => false, 'orderby' => 'name']);
        if (empty($cats)) {
            echo '<tr><td colspan="8">' . esc_html($this->main_instance->__ ('暂无分类。', 'No categories found.')) . '</td></tr>';
        } else {
            foreach ($cats as $cat) {
                echo '<tr>';
                echo '<th class="check-column"><input type="checkbox" name="selected_categories[]" value="' . esc_attr($cat->term_id) . '" /></th>';
                echo '<td>' . esc_html($cat->name) . '</td>';
                echo '<td>' . intval($cat->term_id) . '</td>'; // ✅ ID

                // ✅ 描述列：带 tooltip
                $desc = $cat->description;
                $short_desc = empty($desc) ? '' : wp_trim_words($desc, 20, '…');
                echo '<td>' . esc_html($cat->slug) . '</td>';
                echo '<td>' . intval($cat->count) . '</td>';

                // 父级
                $parent_name = '—';
                if ($cat->parent && $cat->parent !== 0) {
                    $parent_term = get_term($cat->parent, 'category');
                    if ($parent_term && !is_wp_error($parent_term)) {
                        $parent_name = esc_html($parent_term->name);
                    }
                }
                echo '<td>' . $parent_name . '</td>';

                // 描述列 with tooltip
                if (empty($desc)) {
                    echo '<td><em>' . esc_html($this->main_instance->__ ('无', 'None')) . '</em></td>';
                } else {
                    echo '<td class="bcm-description-cell" data-full-desc="' . esc_attr($desc) . '">';
                    echo '<span class="bcm-desc-preview">' . esc_html($short_desc) . '</span>';
                    echo '<div class="bcm-tooltip">' . esc_html($desc) . '</div>';
                    echo '</td>';
                }

                echo '<td>';
                echo '<a href="#" class="edit-category" data-id="' . esc_attr($cat->term_id) . '">' . esc_html($this->main_instance->__ ('快速编辑', 'Quick Edit')) . '</a>';
                echo '</td>';
                echo '</tr>';
            }
        }
        echo '</tbody></table>';

        // 表尾（底部）
        $this->display_tablenav('bottom', $all_cats);
        echo '</form>';

        // ✅ 快速编辑模态框（增强版）
        echo '<div id="edit-category-modal" style="display:none; position:fixed; top:15%; left:50%; transform:translateX(-50%); background:white; padding:20px; border:1px solid #ccc; z-index:9999; box-shadow:0 0 15px rgba(0,0,0,0.4); max-width:600px; width:90%;">';
        echo '<h3>' . esc_html($this->main_instance->__ ('编辑分类', 'Edit Category')) . '</h3>';
        echo '<form id="edit-category-form">';
        echo '<input type="hidden" id="edit-cat-id" name="id" />';
        echo '<p><label>' . esc_html($this->main_instance->__ ('名称', 'Name')) . ':</label><br>';
        echo '<input type="text" id="edit-cat-name" name="name" style="width:100%; margin:5px 0;" required /></p>';
        echo '<p><label>' . esc_html($this->main_instance->__ ('别名', 'Slug')) . ':</label><br>';
        echo '<input type="text" id="edit-cat-slug" name="slug" style="width:100%; margin:5px 0;" /></p>';
        echo '<p><label>' . esc_html($this->main_instance->__ ('父级', 'Parent')) . ':</label><br>';
        echo '<select id="edit-cat-parent" name="parent" style="width:100%; margin:5px 0;"><option value="0">— 无父级 —</option></select></p>';
        echo '<p><label>' . esc_html($this->main_instance->__ ('描述', 'Description')) . ':</label><br>';
        echo '<textarea id="edit-cat-description" name="description" rows="3" style="width:100%; margin:5px 0;"></textarea></p>';
        echo '<button type="submit" class="button button-primary">' . esc_html($this->main_instance->__ ('保存', 'Save')) . '</button> ';
        echo '<button type="button" class="button" onclick="document.getElementById(\'edit-category-modal\').style.display=\'none\'">' . esc_html($this->main_instance->__ ('取消', 'Cancel')) . '</button>';
        echo '</form>';
        echo '</div>';

        // ✅ 添加描述列 Tooltip 的 CSS 和 JS
        ?>
        <style>
        .bcm-description-cell {
            position: relative;
            max-width: 250px;
            cursor: help;
            vertical-align: top;
        }
        .bcm-desc-preview {
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
            text-overflow: ellipsis;
            line-height: 1.4;
        }
        .bcm-tooltip {
            display: none;
            position: absolute;
            top: 100%;
            left: 0;
            background: #23282d;
            color: #fff;
            padding: 8px 10px;
            border-radius: 4px;
            font-size: 12px;
            white-space: pre-wrap;
            word-wrap: break-word;
            max-width: 300px;
            z-index: 100000;
            box-shadow: 0 2px 8px rgba(0,0,0,0.2);
            line-height: 1.4;
        }
        .bcm-tooltip::before {
            content: '';
            position: absolute;
            bottom: 100%;
            left: 10px;
            border-width: 5px;
            border-style: solid;
            border-color: transparent transparent #23282d transparent;
        }
        </style>
        <script>
        jQuery(document).ready(function($) {
            $('.bcm-description-cell').on('mouseenter', function() {
                $(this).find('.bcm-tooltip').show();
            }).on('mouseleave', function() {
                $(this).find('.bcm-tooltip').hide();
            });
        });
        </script>
        <?php

        echo '</div>'; // .wrap
    }

    private function display_tablenav($which, $all_cats) {
        ?>
        <div class="tablenav <?php echo esc_attr($which); ?>">
            <div class="alignleft actions bulkactions">
                <label for="bulk-action-selector-<?php echo esc_attr($which); ?>" class="screen-reader-text">
                    <?php echo esc_html($this->main_instance->__ ('选择批量操作', 'Select bulk action')); ?>
                </label>
                <select name="action" id="bulk-action-selector-<?php echo esc_attr($which); ?>">
                    <option value="-1"><?php echo esc_html($this->main_instance->__ ('批量操作', 'Bulk Actions')); ?></option>
                    <option value="move"><?php echo esc_html($this->main_instance->__ ('移动到', 'Move to')); ?></option>
                    <option value="delete"><?php echo esc_html($this->main_instance->__ ('删除', 'Delete')); ?></option>
                </select>
                <!-- 移动目标分类 -->
                <select name="target_category" style="display:none;" id="target-cat-move-<?php echo esc_attr($which); ?>">
                    <?php foreach ($all_cats as $cat): ?>
                        <option value="<?php echo esc_attr($cat->term_id); ?>"><?php echo esc_html($cat->name); ?></option>
                    <?php endforeach; ?>
                </select>
                <!-- 删除时的文章转移目标 -->
                <select name="target_category_for_delete" style="display:none;" id="target-cat-delete-<?php echo esc_attr($which); ?>">
                    <option value=""><?php echo esc_html($this->main_instance->__ ('— 请选择文章转移目标 —', '— Select target for posts —')); ?></option>
                    <?php foreach ($all_cats as $cat): ?>
                        <option value="<?php echo esc_attr($cat->term_id); ?>"><?php echo esc_html($cat->name); ?></option>
                    <?php endforeach; ?>
                </select>
                <?php submit_button($this->main_instance->__ ('应用', 'Apply'), 'action', '', false); ?>
            </div>
        </div>
        <script>
        (function($) {
            var selector = $('#bulk-action-selector-<?php echo esc_js($which); ?>');
            var moveSelect = $('#target-cat-move-<?php echo esc_js($which); ?>');
            var deleteSelect = $('#target-cat-delete-<?php echo esc_js($which); ?>');
            selector.on('change', function() {
                if (this.value === 'delete') {
                    moveSelect.hide();
                    deleteSelect.show();
                } else if (this.value === 'move') {
                    moveSelect.show();
                    deleteSelect.hide();
                } else {
                    moveSelect.hide();
                    deleteSelect.hide();
                }
            });
            $('#categories-form').on('submit', function(e) {
                if (selector.val() === 'delete') {
                    var target = $('select[name="target_category_for_delete"]').val();
                    if (!target) {
                        alert('<?php echo esc_js($this->main_instance->__ ('请先选择文章要转移的目标分类！', 'Please select a target category for posts!')); ?>');
                        e.preventDefault();
                        return false;
                    }
                }
            });
        })(jQuery);
        </script>
        <?php
    }

    public function handle_bulk_actions() {
        if (!current_user_can('manage_categories')) return;
        $action = sanitize_text_field($_POST['action'] ?? '');
        $selected = isset($_POST['selected_categories']) ? array_map('intval', $_POST['selected_categories']) : [];
        $count = 0;
        $errors = 0;
        if (empty($selected)) return;

        if ($action === 'move') {
            $target_cat_id = intval($_POST['target_category'] ?? 0);
            if (!term_exists($target_cat_id, 'category')) {
                add_settings_error('bcm', 'move_error', $this->main_instance->__ ('目标分类无效。', 'Invalid target category.'), 'error');
                return;
            }
            foreach ($selected as $cat_id) {
                if ($cat_id == $target_cat_id) continue;
                $posts = get_posts(['numberposts' => -1, 'category' => $cat_id, 'post_type' => 'post']);
                foreach ($posts as $post) {
                    $cats = wp_get_post_categories($post->ID);
                    $cats = array_diff($cats, [$cat_id]);
                    $cats[] = $target_cat_id;
                    wp_set_post_categories($post->ID, $cats);
                }
                $count += count($posts);
            }
            add_settings_error('bcm', 'move_complete', sprintf($this->main_instance->__ ('成功移动 %d 篇文章。', 'Successfully moved %d posts.'), $count), 'updated');
        } elseif ($action === 'delete') {
            $target_cat_id = intval($_POST['target_category_for_delete'] ?? 0);
            if (!term_exists($target_cat_id, 'category')) {
                add_settings_error('bcm', 'delete_error', $this->main_instance->__ ('请选择有效的文章转移目标分类。', 'Please select a valid target category for posts.'), 'error');
                return;
            }
            foreach ($selected as $cat_id) {
                if ($cat_id == $target_cat_id) continue;
                if (wp_delete_term($cat_id, 'category', ['default' => $target_cat_id])) {
                    $count++;
                } else {
                    $errors++;
                }
            }
            if ($errors === 0) {
                add_settings_error('bcm', 'delete_complete', sprintf($this->main_instance->__ ('成功删除 %d 个分类，文章已转移。', 'Successfully deleted %d categories, posts transferred.'), $count), 'updated');
            } else {
                add_settings_error('bcm', 'delete_partial', sprintf($this->main_instance->__ ('部分失败：成功删除 %d 个，失败 %d 个。', 'Partially failed: %d deleted, %d failed.'), $count, $errors), 'error');
            }
        }
        $this->main_instance->log_operation('bulk_action_categories', "Action: {$action}, Count: " . count($selected), $errors === 0, '已完成', 0);
    }
}