<?php
if (!defined('ABSPATH')) {
    exit;
}

class Bulk_Content_Manager {

    private $log_option = 'bcm_operation_logs';
    private $export_dir;
    private $language; // 'zh', 'en', or 'auto'

    public function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'handle_import']);
        add_action('admin_init', [$this, 'handle_bulk_actions']);
        add_action('wp_ajax_bcm_update_title', [$this, 'ajax_update_title']);
        add_action('wp_ajax_bcm_update_category', [$this, 'ajax_update_category']);
        add_action('wp_ajax_bcm_get_category', [$this, 'ajax_get_category']);
        add_action('wp_ajax_bcm_update_tag', [$this, 'ajax_update_tag']);
        add_action('wp_ajax_bcm_get_tag', [$this, 'ajax_get_tag']);
        add_action('wp_ajax_bcm_get_post_full', [$this, 'ajax_get_post_full']);
        // ✅ 新增：用于增强版分类快速编辑
        add_action('wp_ajax_bcm_get_category_full', [$this, 'ajax_get_category_full']);
        add_action('wp_ajax_bcm_get_all_categories_for_select', [$this, 'ajax_get_all_categories_for_select']);
        add_action('wp_ajax_bcm_update_category_full', [$this, 'ajax_update_category_full']);
        add_action('admin_init', [$this, 'setup_export_directory']);
        add_action('admin_post_bcm_export_logs', [$this, 'export_logs_csv']);
        add_action('admin_post_bcm_export_categories', [$this, 'export_categories_csv']);
        add_action('admin_post_bcm_export_posts', [$this, 'export_posts_csv']);
        add_action('admin_post_bcm_export_tags', [$this, 'export_tags_csv']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_head', [$this, 'admin_styles']);
    }

    public function enqueue_scripts($hook) {
        // 只在本插件页面加载脚本
        if (strpos($hook, 'bulk-content-manager') !== false) {
            wp_enqueue_script(
                'bcm-admin-js',
                plugin_dir_url(__FILE__) . '../assets/admin.js',
                ['jquery'],
                '2.6.0',
                true
            );
            // 传递 ajaxurl 给 JS
            wp_localize_script('bcm-admin-js', 'bcm_ajax', [
                'ajax_url' => admin_url('admin-ajax.php')
            ]);
        }
    }

    public function admin_styles() {
        echo '<style>
            .bcm-import-export { margin: 1em 0; padding: 12px; background: #f9f9f9; border-left: 4px solid #0073aa; }
            .bcm-import-export input[type="file"] { margin-right: 10px; }
            .bcm-import-export .button { margin-right: 8px; }
            .bcm-thumb-preview { display: inline-block; width: 32px; height: 32px; background-size: cover; background-position: center; border: 1px solid #ddd; margin-right: 5px; cursor: pointer; }
            .bcm-image-count { font-weight: bold; color: #0073aa; }
            /* 批量操作样式 */
            .tablenav.top { margin-bottom: 10px; }
            .tablenav.bottom { margin-top: 10px; }
            /* 修复多选框样式 */
            .wp-list-table .check-column input[type="checkbox"] { margin: 0; }
            /* 新增：transfer 搜索区域 */
            .postbox { padding: 20px; margin-bottom: 25px; background: #f9f9f9; border-radius: 8px; }
        </style>';
    }

    /** * 初始化语言设置 */
    private function init_language() {
        if (isset($_GET['lang']) && in_array($_GET['lang'], ['zh', 'en', 'auto'])) {
            $lang = sanitize_key($_GET['lang']);
            setcookie('bcm_lang', $lang, time() + YEAR_IN_SECONDS, COOKIEPATH ?: '/', '', is_ssl(), true);
            $this->language = $lang;
            return;
        }
        if (isset($_COOKIE['bcm_lang']) && in_array($_COOKIE['bcm_lang'], ['zh', 'en', 'auto'])) {
            $this->language = $_COOKIE['bcm_lang'];
        } else {
            $this->language = 'auto';
        }
        if ($this->language === 'auto') {
            $site_lang = get_locale();
            $this->language = strpos($site_lang, 'zh') === 0 ? 'zh' : 'en';
        }
    }

    public function __(string $zh, string $en): string {
        return $this->language === 'zh' ? $zh : $en;
    }

    public function setup_export_directory() {
        $upload_dir = wp_upload_dir();
        $this->export_dir = '';
        if (empty($upload_dir['error']) && !empty($upload_dir['basedir']) && is_writable($upload_dir['basedir'])) {
            $this->export_dir = $upload_dir['basedir'] . '/bcm-exports';
        } else {
            $temp_dir = sys_get_temp_dir();
            if ($temp_dir && is_writable($temp_dir)) {
                $this->export_dir = $temp_dir . '/bcm-exports';
            } else {
                $this->export_dir = '/tmp/bcm-exports';
            }
        }
        if (!is_dir($this->export_dir)) {
            wp_mkdir_p($this->export_dir);
        }
        if (!is_writable($this->export_dir) && !empty($upload_dir['basedir']) && is_writable($upload_dir['basedir'])) {
            $this->export_dir = $upload_dir['basedir'] . '/bcm-temp';
            wp_mkdir_p($this->export_dir);
        }
        if (!is_writable($this->export_dir)) {
            $plugin_dir = plugin_dir_path(__FILE__) . 'exports';
            $this->export_dir = $plugin_dir;
            wp_mkdir_p($this->export_dir);
        }
    }

    public function add_admin_menu() {
        $this->init_language();
        $menu_title = $this->__ ('批量管理', 'Bulk Mgr');
        add_menu_page(
            $this->__ ('批量内容管理', 'Bulk Content Manager'),
            $menu_title,
            'manage_options',
            'bulk-content-manager',
            [$this, 'dashboard_page'],
            'dashicons-list-view',
            90
        );
    }

    public function dashboard_page() {
        echo '<div class="wrap"><h1>' . esc_html($this->__ ('批量内容管理', 'Bulk Content Manager')) . '</h1>';

        // 语言切换
        $current_url = remove_query_arg(['lang']);
        $langs = [
            'zh' => '中文',
            'en' => 'English',
            'auto' => $this->__ ('跟随站点', 'Follow Site')
        ];
        echo '<div style="position:absolute; top:56px; right:20px;">';
        foreach ($langs as $code => $label) {
            $url = add_query_arg('lang', $code, $current_url);
            $active = ($this->language === $code || ($this->language !== 'zh' && $this->language !== 'en' && $code === 'auto'));
            echo '<a href="' . esc_url($url) . '" class="button' . ($active ? ' button-primary' : '') . '" style="margin-left:5px;">' . esc_html($label) . '</a>';
        }
        echo '</div>';

        $tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'categories';
        ?>
        <h2 class="nav-tab-wrapper">
            <a href="?page=bulk-content-manager&tab=categories" class="nav-tab <?php echo $tab === 'categories' ? 'nav-tab-active' : ''; ?>"><?php echo esc_html($this->__ ('分类管理', 'Categories')); ?></a>
            <a href="?page=bulk-content-manager&tab=posts" class="nav-tab <?php echo $tab === 'posts' ? 'nav-tab-active' : ''; ?>"><?php echo esc_html($this->__ ('文章管理', 'Posts')); ?></a>
            <a href="?page=bulk-content-manager&tab=tags" class="nav-tab <?php echo $tab === 'tags' ? 'nav-tab-active' : ''; ?>"><?php echo esc_html($this->__ ('标签管理', 'Tags')); ?></a>
            <!-- ✅ 批量转移：移到 Logs 前 -->
            <a href="?page=bulk-content-manager&tab=transfer" class="nav-tab <?php echo $tab === 'transfer' ? 'nav-tab-active' : ''; ?>">
                <?php echo esc_html($this->__ ('批量转移', 'Transfer')); ?></a>
            <a href="?page=bulk-content-manager&tab=logs" class="nav-tab <?php echo $tab === 'logs' ? 'nav-tab-active' : ''; ?>">
                <?php echo esc_html($this->__ ('操作记录', 'Logs')); ?></a>
        </h2>
        <?php
        if ($tab === 'categories') {
            $handler = new Categories_Handler($this);
            $handler->render_page();
        } elseif ($tab === 'posts') {
            $handler = new Posts_Handler($this);
            $handler->render_page();
        } elseif ($tab === 'tags') {
            $handler = new Tags_Handler($this);
            $handler->render_page();
        } elseif ($tab === 'logs') {
            $handler = new Logs_Handler($this);
            $handler->render_page();
        }
        // ✅ 新增：transfer tab 渲染
        elseif ($tab === 'transfer') {
            $handler = new Transfer_Handler($this);
            $handler->render_page();
        }
        echo '<p style="margin-top:40px; padding-top:15px; border-top:1px solid #eee; color:#777; font-size:13px;">';
        echo '&copy; 2026 <a href="https://limaomao810.com" target="_blank" rel="noopener">limaomao810.com</a>';
        echo '</p>';
        echo '</div>';
    }

    // ================== AJAX 方法 ==================
    public function ajax_update_title() {
        if (!current_user_can('edit_posts')) wp_die('Unauthorized');
        $id = intval($_POST['id']);
        $title = sanitize_text_field($_POST['title']);
        if ($id && $title) {
            wp_update_post(['ID' => $id, 'post_title' => $title]);
            $this->log_operation('update_title', "Post ID: {$id}, New Title: {$title}", true, '已完成', 0);
        }
        wp_die();
    }

    public function ajax_update_category() {
        if (!current_user_can('manage_categories')) wp_die('Unauthorized');
        $id = intval($_POST['id']);
        $name = sanitize_text_field($_POST['name']);
        if ($id && $name) {
            wp_update_term($id, 'category', ['name' => $name]);
            $this->log_operation('update_category', "Category ID: {$id}, New Name: {$name}", true, '已完成', 0);
        }
        wp_die();
    }

    public function ajax_get_category() {
        if (!current_user_can('manage_categories')) wp_die('Unauthorized');
        $id = intval($_GET['id']);
        $term = get_term($id, 'category');
        if ($term && !is_wp_error($term)) {
            wp_send_json(['term_id' => $term->term_id, 'name' => $term->name]);
        }
        wp_send_json([]);
    }

    public function ajax_update_tag() {
        if (!current_user_can('manage_categories')) wp_die('Unauthorized');
        $id = intval($_POST['id']);
        $name = sanitize_text_field($_POST['name']);
        if ($id && $name) {
            wp_update_term($id, 'post_tag', ['name' => $name]);
            $this->log_operation('update_tag', "Tag ID: {$id}, New Name: {$name}", true, '已完成', 0);
        }
        wp_die();
    }

    public function ajax_get_tag() {
        if (!current_user_can('manage_categories')) wp_die('Unauthorized');
        $id = intval($_GET['id']);
        $term = get_term($id, 'post_tag');
        if ($term && !is_wp_error($term)) {
            wp_send_json(['term_id' => $term->term_id, 'name' => $term->name]);
        }
        wp_send_json([]);
    }

    public function ajax_get_post_full() {
        if (!current_user_can('edit_posts')) wp_die('Unauthorized');
        $id = intval($_GET['id']);
        $post = get_post($id);
        if ($post) {
            wp_send_json(['ID' => $post->ID, 'post_title' => $post->post_title]);
        }
        wp_send_json([]);
    }

    // ================== 新增 AJAX 方法：增强分类编辑 ==================

    /**
     * 获取分类完整信息（用于快速编辑模态框）
     */
    public function ajax_get_category_full() {
        if (!current_user_can('manage_categories')) {
            wp_die('Unauthorized');
        }
        $id = intval($_POST['id'] ?? 0);
        if (!$id) {
            wp_send_json_error('Invalid ID');
        }
        $term = get_term($id, 'category');
        if ($term && !is_wp_error($term)) {
            wp_send_json_success([
                'term_id' => $term->term_id,
                'name' => $term->name,
                'slug' => $term->slug,
                'description' => $term->description,
                'parent' => (int)$term->parent
            ]);
        }
        wp_send_json_error('Category not found');
    }

    /**
     * 获取所有分类（用于父级下拉选择，可排除自身）
     */
    public function ajax_get_all_categories_for_select() {
        if (!current_user_can('manage_categories')) {
            wp_die('Unauthorized');
        }
        $exclude = intval($_POST['exclude'] ?? 0);
        $categories = get_categories([
            'hide_empty' => false,
            'orderby' => 'name',
            'fields' => 'ids'
        ]);
        $result = [];
        foreach ($categories as $cat_id) {
            if ($cat_id == $exclude) continue;
            $term = get_term($cat_id, 'category');
            if ($term && !is_wp_error($term)) {
                $result[] = [
                    'term_id' => $term->term_id,
                    'name' => $term->name
                ];
            }
        }
        wp_send_json_success($result);
    }

    /**
     * 保存完整的分类信息
     */
    public function ajax_update_category_full() {
        if (!current_user_can('manage_categories')) {
            wp_die('Unauthorized');
        }
        $id = intval($_POST['id'] ?? 0);
        $name = sanitize_text_field($_POST['name'] ?? '');
        $slug = sanitize_title($_POST['slug'] ?: $name);
        $parent = intval($_POST['parent'] ?? 0);
        $description = sanitize_textarea_field($_POST['description'] ?? '');

        if (!$id || !$name) {
            wp_send_json_error('Missing required fields');
        }

        // 验证父级是否存在（若指定）
        if ($parent > 0 && !term_exists($parent, 'category')) {
            wp_send_json_error('Invalid parent category');
        }

        $update = wp_update_term($id, 'category', [
            'name' => $name,
            'slug' => $slug,
            'description' => $description,
            'parent' => $parent
        ]);

        if (!is_wp_error($update)) {
            $this->log_operation('update_category_full', "Category ID: {$id}, Name: {$name}", true, '已完成', 0);
            wp_send_json_success();
        } else {
            wp_send_json_error($update->get_error_message());
        }
    }

    // ================== 日志记录 ==================
    public function log_operation($action, $details, $success, $status, $duration) {
        // ✅ 获取当前操作人
        $user = wp_get_current_user();
        $operator = $user && $user->exists() ? $user->user_login : 'system';

        $logs = get_option($this->log_option, []);
        $logs[] = [
            'timestamp' => time(),
            'action' => $action,
            'details' => substr($details, 0, 500),
            'success' => (bool)$success,
            'status' => $status,
            'duration' => (float)$duration,
            'operator' => $operator, // ✅ 新增字段
        ];
        if (count($logs) > 200) {
            $logs = array_slice($logs, -200);
        }
        update_option($this->log_option, $logs, false);
    }

    // ================== 导入处理 ==================
    public function handle_import() {
        if (!current_user_can('manage_options')) return;
        if (isset($_POST['import_categories']) && wp_verify_nonce($_POST['bcm_nonce'], 'bcm_import_categories')) {
            $this->import_categories_from_csv($_FILES['csv_file']);
        } elseif (isset($_POST['import_posts']) && wp_verify_nonce($_POST['bcm_nonce'], 'bcm_import_posts')) {
            $this->import_posts_from_csv($_FILES['csv_file']);
        } elseif (isset($_POST['import_tags']) && wp_verify_nonce($_POST['bcm_nonce'], 'bcm_import_tags')) {
            $this->import_tags_from_csv($_FILES['csv_file']);
        }
    }

    private function import_categories_from_csv($file) {
        $start_time = microtime(true);
        if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
            add_settings_error('bcm', 'import_error', $this->__ ('文件上传失败。', 'File upload failed.'), 'error');
            return;
        }
        $handle = fopen($file['tmp_name'], 'r');
        if (!$handle) {
            add_settings_error('bcm', 'import_error', $this->__ ('无法读取 CSV 文件。', 'Cannot read CSV file.'), 'error');
            return;
        }
        $success = 0;
        $fail = 0;
        fgetcsv($handle); // 跳过标题行
        while (($data = fgetcsv($handle)) !== false) {
            if (count($data) < 2) continue;
            $name = sanitize_text_field($data[0]);
            $slug = sanitize_title($data[1] ?: $name);
            if (!term_exists($slug, 'category')) {
                $result = wp_insert_term($name, 'category', ['slug' => $slug]);
                if (!is_wp_error($result)) {
                    $success++;
                } else {
                    $fail++;
                }
            }
        }
        fclose($handle);
        unlink($file['tmp_name']);
        $message = sprintf($this->__ ('成功导入 %d 个分类，跳过/失败 %d 个。', 'Successfully imported %d categories, skipped/failed %d.'), $success, $fail);
        add_settings_error('bcm', 'import_complete', $message, $fail > 0 ? 'error' : 'updated');
        $this->log_operation('import_categories', "File: {$file['name']}", true, '已完成', microtime(true) - $start_time);
    }

    private function import_posts_from_csv($file) {
        $start_time = microtime(true);
        if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
            add_settings_error('bcm', 'import_error', $this->__ ('文件上传失败。', 'File upload failed.'), 'error');
            return;
        }
        $handle = fopen($file['tmp_name'], 'r');
        if (!$handle) {
            add_settings_error('bcm', 'import_error', $this->__ ('无法读取 CSV 文件。', 'Cannot read CSV file.'), 'error');
            return;
        }
        $success = 0;
        $fail = 0;
        fgetcsv($handle); // 跳过标题行
        while (($data = fgetcsv($handle)) !== false) {
            if (count($data) < 2) continue;
            $post_id = intval($data[0]);
            $new_title = sanitize_text_field($data[1]);
            if ($post_id && $new_title && get_post($post_id)) {
                $result = wp_update_post(['ID' => $post_id, 'post_title' => $new_title], true);
                if (!is_wp_error($result)) {
                    $success++;
                } else {
                    $fail++;
                }
            }
        }
        fclose($handle);
        unlink($file['tmp_name']);
        $message = sprintf($this->__ ('成功更新 %d 篇文章标题，失败 %d 篇。', 'Successfully updated %d post titles, failed %d.'), $success, $fail);
        add_settings_error('bcm', 'import_complete', $message, $fail > 0 ? 'error' : 'updated');
        $this->log_operation('import_posts', "File: {$file['name']}", true, '已完成', microtime(true) - $start_time);
    }

    private function import_tags_from_csv($file) {
        $start_time = microtime(true);
        if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
            add_settings_error('bcm', 'import_error', $this->__ ('文件上传失败。', 'File upload failed.'), 'error');
            return;
        }
        $handle = fopen($file['tmp_name'], 'r');
        if (!$handle) {
            add_settings_error('bcm', 'import_error', $this->__ ('无法读取 CSV 文件。', 'Cannot read CSV file.'), 'error');
            return;
        }
        $success = 0;
        $fail = 0;
        fgetcsv($handle); // 跳过标题行
        while (($data = fgetcsv($handle)) !== false) {
            if (count($data) < 2) continue;
            $name = sanitize_text_field($data[0]);
            $slug = sanitize_title($data[1] ?: $name);
            if (!term_exists($slug, 'post_tag')) {
                $result = wp_insert_term($name, 'post_tag', ['slug' => $slug]);
                if (!is_wp_error($result)) {
                    $success++;
                } else {
                    $fail++;
                }
            }
        }
        fclose($handle);
        unlink($file['tmp_name']);
        $message = sprintf($this->__ ('成功导入 %d 个标签，跳过/失败 %d 个。', 'Successfully imported %d tags, skipped/failed %d.'), $success, $fail);
        add_settings_error('bcm', 'import_complete', $message, $fail > 0 ? 'error' : 'updated');
        $this->log_operation('import_tags', "File: {$file['name']}", true, '已完成', microtime(true) - $start_time);
    }

    // ================== 批量操作处理 ==================
    public function handle_bulk_actions() {
        $tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'categories';
        if ($tab === 'posts' && isset($_POST['action']) && $_POST['action'] !== '-1') {
            $handler = new Posts_Handler($this);
            $handler->handle_bulk_actions();
        } elseif ($tab === 'tags' && isset($_POST['action']) && $_POST['action'] !== '-1') {
            $handler = new Tags_Handler($this);
            $handler->handle_bulk_actions();
        } elseif ($tab === 'categories' && isset($_POST['action']) && $_POST['action'] !== '-1') {
            $handler = new Categories_Handler($this);
            $handler->handle_bulk_actions();
        }
        // ✅ 新增：处理 transfer 提交
        elseif ($tab === 'transfer') {
            $handler = new Transfer_Handler($this);
            $handler->handle_transfer();
        }
    }

    // ================== 导出 URL 生成 ==================
    public function get_export_url($type) {
        $actions = [
            'posts' => 'bcm_export_posts',
            'categories' => 'bcm_export_categories',
            'tags' => 'bcm_export_tags',
            'logs' => 'bcm_export_logs'
        ];
        return admin_url('admin-post.php?action=' . $actions[$type]);
    }

    // ================== 导出方法 ==================
    private function export_csv($filename, $headers, $rows) {
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        $output = fopen('php://output', 'w');
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF)); // UTF-8 BOM
        fputcsv($output, $headers);
        foreach ($rows as $row) {
            fputcsv($output, $row);
        }
        fclose($output);
        exit;
    }

    public function export_posts_csv() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        $posts = get_posts(['numberposts' => -1, 'post_type' => 'post']);
        $rows = [];
        foreach ($posts as $post) {
            $cats = get_the_category($post->ID);
            $cat_names = $cats ? implode(', ', wp_list_pluck($cats, 'name')) : '';
            $tags = get_the_tags($post->ID);
            $tag_names = $tags ? implode(', ', wp_list_pluck($tags, 'name')) : '';
            $thumb_id = get_post_thumbnail_id($post->ID);
            $has_thumb = $thumb_id ? '是' : '否';
            $rows[] = [
                $post->ID,
                $post->post_title,
                $cat_names,
                $tag_names,
                $has_thumb,
                $post->post_status,
                get_the_date('Y-m-d H:i:s', $post->ID)
            ];
        }
        $this->export_csv('posts_export_' . date('Ymd_His') . '.csv', ['ID', '标题', '分类', '标签', '有特色图像', '状态', '发布日期'], $rows);
    }

    public function export_categories_csv() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        $cats = get_categories(['hide_empty' => false]);
        $rows = [];
        foreach ($cats as $cat) {
            $parent_name = $cat->parent ? get_term($cat->parent, 'category')->name : '';
            $rows[] = [$cat->term_id, $cat->name, $cat->slug, $parent_name, $cat->count];
        }
        $this->export_csv('categories_export_' . date('Ymd_His') . '.csv', ['ID', '名称', '别名', '父级', '文章数'], $rows);
    }

    public function export_tags_csv() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        $tags = get_terms(['taxonomy' => 'post_tag', 'hide_empty' => false]);
        $rows = [];
        foreach ($tags as $tag) {
            $rows[] = [$tag->term_id, $tag->name, $tag->slug, $tag->count];
        }
        $this->export_csv('tags_export_' . date('Ymd_His') . '.csv', ['ID', '名称', '别名', '文章数'], $rows);
    }

    public function export_logs_csv() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        $logs = get_option($this->log_option, []);
        $rows = [];
        foreach ($logs as $log) {
            $rows[] = [
                date('Y-m-d H:i:s', bcm_get_log_field($log, 'timestamp', 0)),
                bcm_get_log_field($log, 'action', ''),
                bcm_get_log_field($log, 'details', ''),
                bcm_get_log_field($log, 'success', false) ? '是' : '否',
                bcm_get_log_field($log, 'status', ''),
                number_format(bcm_get_log_field($log, 'duration', 0), 2),
                bcm_get_log_field($log, 'operator', 'system') // ✅ 导出操作人
            ];
        }
        $this->export_csv('operation_logs_' . date('Ymd_His') . '.csv', ['时间', '操作', '详情', '成功', '状态', '耗时(秒)', '操作人'], $rows);
    }
}